﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kms/KMSRequest.h>
#include <aws/kms/KMS_EXPORTS.h>

#include <utility>

namespace Aws {
namespace KMS {
namespace Model {

/**
 */
class DeleteImportedKeyMaterialRequest : public KMSRequest {
 public:
  AWS_KMS_API DeleteImportedKeyMaterialRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteImportedKeyMaterial"; }

  AWS_KMS_API Aws::String SerializePayload() const override;

  AWS_KMS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Identifies the KMS key from which you are deleting imported key material. The
   * <code>Origin</code> of the KMS key must be <code>EXTERNAL</code>.</p> <p>Specify
   * the key ID or key ARN of the KMS key.</p> <p>For example:</p> <ul> <li> <p>Key
   * ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code> </p> </li> <li> <p>Key
   * ARN:
   * <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
   * </p> </li> </ul> <p>To get the key ID and key ARN for a KMS key, use
   * <a>ListKeys</a> or <a>DescribeKey</a>.</p>
   */
  inline const Aws::String& GetKeyId() const { return m_keyId; }
  inline bool KeyIdHasBeenSet() const { return m_keyIdHasBeenSet; }
  template <typename KeyIdT = Aws::String>
  void SetKeyId(KeyIdT&& value) {
    m_keyIdHasBeenSet = true;
    m_keyId = std::forward<KeyIdT>(value);
  }
  template <typename KeyIdT = Aws::String>
  DeleteImportedKeyMaterialRequest& WithKeyId(KeyIdT&& value) {
    SetKeyId(std::forward<KeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Identifies the imported key material you are deleting. </p>  <p>If
   * no KeyMaterialId is specified, KMS deletes the current key material.</p>
   *  <p>To get the list of key material IDs associated with a KMS key,
   * use <a>ListKeyRotations</a>.</p>
   */
  inline const Aws::String& GetKeyMaterialId() const { return m_keyMaterialId; }
  inline bool KeyMaterialIdHasBeenSet() const { return m_keyMaterialIdHasBeenSet; }
  template <typename KeyMaterialIdT = Aws::String>
  void SetKeyMaterialId(KeyMaterialIdT&& value) {
    m_keyMaterialIdHasBeenSet = true;
    m_keyMaterialId = std::forward<KeyMaterialIdT>(value);
  }
  template <typename KeyMaterialIdT = Aws::String>
  DeleteImportedKeyMaterialRequest& WithKeyMaterialId(KeyMaterialIdT&& value) {
    SetKeyMaterialId(std::forward<KeyMaterialIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_keyId;
  bool m_keyIdHasBeenSet = false;

  Aws::String m_keyMaterialId;
  bool m_keyMaterialIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace KMS
}  // namespace Aws
